﻿using System;
using System.Runtime.InteropServices;
using System.Security.AccessControl;
using System.Text;

namespace Synexens
{
    public static class SYSDKInterface
    {
        private const string SYSDK_DLL = "SynexensSDK.dll"; 

        // 获取SDK版本
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode GetSDKVersion(ref int nLength, StringBuilder strSDKVersion);

        // 初始化SDK
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode InitSDK();

        // 反初始化SDK，释放资源
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode UnInitSDK();
        
        //查找设备
        //@ param [in/out] nCount 设备数量
        //@ param [in/out] pDevice 设备信息，由外部分配内存，pDevice传入nullptr时仅获取nCount
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode FindDevice(out int nCount, [In, Out] SYDeviceInfo[] pDevice);
        
        //打开设备
        //@ param [in] deviceInfo 设备信息
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode OpenDevice(ref SYDeviceInfo deviceInfo);
        
        //关闭设备
        //@ param [in] nDeviceID 设备ID
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode CloseDevice(uint nDeviceID);
        
        //查询设备支持类型
        //@ param [in] nDeviceID 设备ID
        //@ param [in/out] nCount 支持的数据帧类型数量,pSupportType为空时仅用作返回数量，否则用来校验pSupportType内存分配数量是否匹配
        //@ param [in/out] pSupportType 支持的类型，由外部分配内存，pSupportType传入nullptr时仅获取nCount
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode QueryDeviceSupportFrameType(uint nDeviceID, ref int nCount, [In, Out] SYSupportType[] pSupportType);
        
        //查询设备支持的帧分辨率
        //@ param [in] nDeviceID 设备ID
        //@ param [in] supportType 帧类型
        //@ param [in/out] nCount 支持的分辨率数量,pResolution为空时仅用作返回数量，否则用来校验pResolution内存分配数量是否匹配
        //@ param [in/out] pResolution 支持的分辨率类型，由外部分配内存，pResolution传入nullptr时仅获取nCount
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode QueryDeviceSupportResolution(uint nDeviceID, SYSupportType supportType, ref int nCount, SYResolution[] pResolution);
        
        //获取当前流类型
        //@ param [in] nDeviceID 设备ID
        //@ return 当前流类型
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYStreamType GetCurrentStreamType(uint nDeviceID);
        
        //启动数据流
        //@ param [in] nDeviceID 设备ID
        //@ param [in] streamType 数据流类型
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode StartStreaming(uint nDeviceID, SYStreamType streamType);
        
        //停止数据流
        //@ param [in] nDeviceID 设备ID
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode StopStreaming(uint nDeviceID);
        
        //切换数据流
        //@ param [in] nDeviceID 设备ID
        //@ param [in] streamType 数据流类型
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode ChangeStreaming(uint nDeviceID, SYStreamType streamType);
        
        //设置分辨率（如果已启动数据流，内部会执行关流->设置分辨率->重新开流的操作流程）
        //@ param [in] nDeviceID 设备ID
        //@ param [in] frameType 帧类型
        //@ param [in] resolution 帧分辨率
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode SetFrameResolution(uint nDeviceID, SYFrameType frameType, SYResolution resolution);
        
        //获取设备帧分辨率
        //@ param [in] nDeviceID 设备ID
        //@ param [in] frameType 帧类型
        //@ param [out] resolution 帧分辨率
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode GetFrameResolution(uint nDeviceID, SYFrameType frameType, ref SYResolution resolution);
        
        //获取滤波开启状态
        //@ param [in] nDeviceID 设备ID
        //@ param [out] bFilter 滤波开启状态，true-已开启滤波，false-未开启滤波
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode GetFilter(uint nDeviceID, ref bool bFilter);
        
        //开启/关闭滤波
        //@ param [in] nDeviceID 设备ID
        //@ param [in] bFilter 滤波开关，true-开启滤波，false-关闭滤波
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode SetFilter(uint nDeviceID, bool bFilter);
        
        //获取硬件滤波模式开启状态
        //@ param [in] nDeviceID 设备ID
        //@ param [out] bHardwareFilterMode 硬件滤波模式开启状态，true-已开启硬件滤波模式，false-未开启硬件滤波模式
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode GetHardWareFilterMode(uint nDeviceID, ref bool bHardwareFilterMode);
        
        //开启/关闭硬件滤波模式
        //@ param [in] nDeviceID 设备ID
        //@ param [in] bHardwareFilterMode 硬件滤波模式开启状态，true-开启硬件滤波模式，false-关闭硬件滤波模式
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode SetHardWareFilterMode(uint nDeviceID, bool bHardwareFilterMode);
        
        //硬件滤波模式是否可用
        //@ param [in] nDeviceID 设备ID
        //@ param [in] bHaveHardwareFilterMode 硬件滤波模式可用标志，true-可用，false-不可用
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode HaveHardWareFilterMode(uint nDeviceID, ref bool bHardwareFilterMode);
        
        //获取滤波列表
        //@ param [in] nDeviceID 设备ID
        //@ param [in/out] nCount 滤波列表长度
        //@ param [in/out] pFilterType 滤波列表
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode GetFilterList(uint nDeviceID, ref int nCount, SYFilterType[] pFilterType);
        
        //设置默认滤波
        //@ param [in] nDeviceID 设备ID
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode SetDefaultFilter(uint nDeviceID);
        
        //增加滤波
        //@ param [in] nDeviceID 设备ID
        //@ param [in] filterType 滤波类型
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode AddFilter(uint nDeviceID, SYFilterType filterType);
        
        //移除滤波
        //@ param [in] nDeviceID 设备ID
        //@ param [in] nIndex 滤波列表中的索引
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode DeleteFilter(uint nDeviceID, int nIndex);
        
        //清除滤波
        //@ param [in] nDeviceID 设备ID
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode ClearFilter(uint nDeviceID);
        
        //设置滤波参数
        //@ param [in] nDeviceID 设备ID
        //@ param [in] filterType 滤波类型
        //@ param [in] nParamCount 滤波参数个数
        //@ param [in] pFilterParam 滤波参数
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode SetFilterParam(uint nDeviceID, SYFilterType filterType, int nParamCount, IntPtr pFilterParam);
        
        //获取滤波参数
        //@ param [in] nDeviceID 设备ID
        //@ param [in] filterType 滤波类型
        //@ param [in/out] nParamCount 滤波参数个数
        //@ param [in/out] pFilterParam 滤波参数
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode GetFilterParam(uint nDeviceID, SYFilterType filterType, ref int nParamCount, IntPtr pFilterParam);
        
        //获取水平镜像状态
        //@ param [in] nDeviceID 设备ID
        //@ param [out] bMirror 水平镜像状态，true-已开启水平镜像，false-未开启水平镜像
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode GetMirror(uint nDeviceID, ref bool bMirror);
        
        //开启/关闭水平镜像
        //@ param [in] nDeviceID 设备ID
        //@ param [in] bMirror 水平镜像开关，true-开启水平镜像，false-关闭水平镜像
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode SetMirror(uint nDeviceID, bool bMirror);
        
        //获取垂直翻转状态
        //@ param [in] nDeviceID 设备ID
        //@ param [out] bFlip 垂直翻转状态，true-已开启垂直翻转，false-未开启垂直翻转
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode GetFlip(uint nDeviceID, ref bool bFlip);
        
        //开启/关闭垂直翻转
        //@ param [in] nDeviceID 设备ID
        //@ param [in] bFlip 垂直翻转开关，true-开启垂直翻转，false-关闭垂直翻转
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode SetFlip(uint nDeviceID, bool bFlip);
        
        //获取积分时间
        //@ param [in] nDeviceID 设备ID
        //@ param [out] nIntegralTime 积分时间
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode GetIntegralTime(uint nDeviceID, ref int nIntegralTime);
        
        //设置积分时间
        //@ param [in] nDeviceID 设备ID
        //@ param [in] nIntegralTime 积分时间
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode SetIntegralTime(uint nDeviceID, int nIntegralTime);
        
        //获取积分时间调节范围
        //@ param [in] nDeviceID 设备ID
        //@ param [in] depthResolution depth分辨率
        //@ param [out] nMin 积分时间最小值
        //@ param [out] nMax 积分时间最大值
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode GetIntegralTimeRange(uint nDeviceID, SYResolution depthResolution, ref int nMin, ref int nMax);
        
        //获取测距量程
        //@ param [in] nDeviceID 设备ID
        //@ param [out] nMin 量程最小值
        //@ param [out] nMax 量程最大值
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode GetDistanceMeasureRange(uint nDeviceID, ref int nMin, ref int nMax);
        
        //获取用户测距范围
        //@ param [in] nDeviceID 设备ID
        //@ param [out] nMin 测距范围最小值
        //@ param [out] nMax 测距范围最大值
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode GetDistanceUserRange(uint nDeviceID, ref int nMin, ref int nMax);
        
        //设置用户测距范围
        //@ param [in] nDeviceID 设备ID
        //@ param [in] nMin 测距范围最小值
        //@ param [in] nMax 测距范围最大值
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode SetDistanceUserRange(uint nDeviceID, int nMin, int nMax);
        
        //读取设备sn号
        //@ param [in] nDeviceID 设备ID
        //@ param [in/out] nLength 字符长度
        //@ param [in/out] pstrSN 设备sn号字符串指针,由外部分配内存，pstrSN传入nullptr时仅获取nLength
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode GetDeviceSN(uint nDeviceID, ref int nLength, StringBuilder strSN);
        
        //写入设备sn号
        //@ param [in] nDeviceID 设备ID
        //@ param [in] nLength 字符长度
        //@ param [in] pstrSN 设备sn号字符串指针
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode SetDeviceSN(uint nDeviceID, int nLength, string pstrSN);
        
        //读取设备固件版本号
        //@ param [in] nDeviceID 设备ID
        //@ param [in/out] nLength 字符长度
        //@ param [in/out] pstrHWVersion 固件版本号字符串指针,由外部分配内存，pstrHWVersion传入nullptr时仅获取nLength
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode GetDeviceHWVersion(uint nDeviceID, ref int nLength, StringBuilder strHWVersion);
        
        //获取深度对应伪彩色
        //@ param [in] nDeviceID 设备ID
        //@ param [in] nCount 数据量(内存空间pDepth需要nCount*2字节，pColor需要nCount*3字节)
        //@ param [in] pDepth 深度数据
        //@ param [in/out] pColor 深度对应伪彩色(24位RGB格式)
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode GetDepthColor(uint nDeviceID, int nCount, IntPtr pDepth, byte[] pColor);
        
        //获取深度对应点云数据
        //@ param [in] nDeviceID 设备ID
        //@ param [in] nWidth 宽度
        //@ param [in] nHeight 高度
        //@ param [in] pDepth 深度数据
        //@ param [in/out] pPointCloud 深度对应点云数据,由外部分配内存
        //@ param [in] bUndistort 裁剪标志，true-裁剪 false-不裁剪
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern unsafe SYErrorCode GetDepthPointCloud(uint nDeviceID, int nWidth, int nHeight, IntPtr pDepth, IntPtr pPointCloud, bool bUndistort = false);
        
        //获取RGBD
        //@ param [in] nDeviceID 设备ID
        //@ param [in] nSourceDepthWidth 源深度数据宽度
        //@ param [in] nSourceDepthHeight 源深度数据高度
        //@ param [in] pSourceDepth 源深度数据
        //@ param [in] nSourceRGBWidth 源RGB数据宽度
        //@ param [in] nSourceRGBHeight 源RGB数据高度
        //@ param [in] pSourceRGB 源RGB数据
        //@ param [in] nTargetWidth RGBD数据宽度
        //@ param [in] nTargetHeight RGBD数据高度
        //@ param [in/out] pTargetDepth RGBD中的深度数据,由外部分配内存,数据长度与源RGB长度一致
        //@ param [in/out] pTargetRGB RGBD中的RGB数据,由外部分配内存,数据长度与源RGB长度一致
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern unsafe SYErrorCode GetRGBD(uint nDeviceID, int nSourceDepthWidth, int nSourceDepthHeight, ushort* pSourceDepth, int nSourceRGBWidth, int nSourceRGBHeight,  byte* pSourceRGB, int nTargetWidth, int nTargetHeight,  IntPtr pTargetDepth,  IntPtr pTargetRGB);
        
        //获取最新一帧数据
        //@ param [in] nDeviceID 设备ID
        //@ param [in/out] pFrameData 最后一帧数据
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode GetLastFrameData(uint nDeviceID, ref IntPtr pFrameData);
        
        //去畸变
        //@ param [in] nDeviceID 设备ID
        //@ param [in] pSource  待去畸变数据指针
        //@ param [in] nWidth 图像宽度
        //@ param [in] nHeight 图像高度
        //@ param [in] bDepth 是否是深度数据/RGB数据
        //@ param [out] pTarget  去畸变结果数据指针，由外部分配内存,数据长度与待去畸变数据指针长度一致
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern unsafe SYErrorCode Undistort(uint nDeviceID, ushort* pSource, int nWidth, int nHeight, bool bDepth, IntPtr pTarget);
        
        //获取相机参数
        //@ param [in] nDeviceID 设备ID
        //@ param [in] resolution  分辨率
        //@ param [in/out] intrinsics 相机参数
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode GetIntric(uint nDeviceID, SYResolution resolution, ref SYIntrinsics intrinsics);
        
        //获取拖影滤波开启状态
        //@ param [in] nDeviceID 设备ID
        //@ param [out] bFilter 拖影滤波开启状态，true-已开启滤波，false-未开启滤波
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode GetTrailFilter(uint nDeviceID, ref bool bFilter);
        
        //开启/关闭拖影滤波
        //@ param [in] nDeviceID 设备ID
        //@ param [in] bFilter 拖影滤波开关，true-开启滤波，false-关闭滤波
        //@ return 错误码
        [DllImport(SYSDK_DLL, CallingConvention = CallingConvention.Cdecl)]
        public static extern SYErrorCode SetTrailFilter(uint nDeviceID, bool bFilter);
    }
}